<?php
// Script to verify that all 22 email templates are properly integrated

// Include test configuration
require_once 'test_mail_config.php';

// Include only the mail system
require_once 'mail_system/integrated_mail_system.php';

echo "📧 ZForex Mail System - Template Integration Verification\n";
echo "========================================================\n\n";

// Get mail system instance
$mailSystem = getMailSystem();

// Check if all templates are loaded
$reflection = new ReflectionClass($mailSystem);
$templatesProperty = $reflection->getProperty('templates');
$templatesProperty->setAccessible(true);
$templates = $templatesProperty->getValue($mailSystem);

echo "Total templates loaded: " . count($templates) . "\n";
echo "Expected templates: 22\n";
echo "Integration status: " . (count($templates) == 22 ? "✅ COMPLETE" : "❌ INCOMPLETE") . "\n\n";

// List all templates
echo "📋 Template List:\n";
echo "================\n";
foreach ($templates as $templateName => $templateData) {
    echo "✅ $templateName\n";
}

echo "\n🔍 Template Structure Verification:\n";
echo "==================================\n";

$requiredFields = ['subject', 'header', 'body'];
$missingTemplates = [];
$incompleteTemplates = [];

foreach ($templates as $templateName => $templateData) {
    $missingFields = [];
    foreach ($requiredFields as $field) {
        if (!isset($templateData[$field])) {
            $missingFields[] = $field;
        }
    }
    
    if (!empty($missingFields)) {
        $incompleteTemplates[$templateName] = $missingFields;
    }
}

if (empty($incompleteTemplates)) {
    echo "✅ All templates have complete structure\n";
} else {
    echo "❌ Incomplete templates:\n";
    foreach ($incompleteTemplates as $templateName => $missingFields) {
        echo "  - $templateName: missing " . implode(', ', $missingFields) . "\n";
    }
}

echo "\n🧪 Quick Send Method Verification:\n";
echo "==================================\n";

// Check if all quick send methods exist
$quickSendMethods = [
    'sendRegisterVerification',
    'sendAfterVerifyCredentials',
    'sendLoginAlert',
    'sendLiveAccountOpened',
    'sendKYCUpload',
    'sendKYCApproved',
    'sendDepositRequestSubmitted',
    'sendDepositApproved',
    'sendDepositFailed',
    'sendWithdrawalOTP',
    'sendWithdrawalRequestSubmitted',
    'sendWithdrawalApproved',
    'sendWithdrawalRejected',
    'sendIBCommissionWithdrawal',
    'sendReferralRegistration',
    'sendTradeOpened',
    'sendTradeClosed',
    'sendDailyTradeSummary',
    'sendPasswordReset',
    'sendPasswordChanged',
    'sendProfileUpdated',
    'sendMonthlyStatement'
];

$missingMethods = [];
foreach ($quickSendMethods as $method) {
    if (!method_exists($mailSystem, $method)) {
        $missingMethods[] = $method;
    }
}

if (empty($missingMethods)) {
    echo "✅ All 22 quick send methods are implemented\n";
} else {
    echo "❌ Missing methods:\n";
    foreach ($missingMethods as $method) {
        echo "  - $method\n";
    }
}

echo "\n🔧 Configuration Verification:\n";
echo "=============================\n";

$requiredConstants = [
    'MAIL_HOST',
    'MAIL_PORT',
    'MAIL_USERNAME',
    'MAIL_PASSWORD',
    'MAIL_ENCRYPTION',
    'MAIL_FROM_EMAIL',
    'MAIL_FROM_NAME',
    'MAIL_LOGO_URL',
    'MAIL_COMPANY_NAME',
    'MAIL_COMPANY_ADDRESS',
    'MAIL_SUPPORT_EMAIL',
    'MAIL_WEBSITE_URL'
];

$missingConstants = [];
foreach ($requiredConstants as $constant) {
    if (!defined($constant)) {
        $missingConstants[] = $constant;
    }
}

if (empty($missingConstants)) {
    echo "✅ All required configuration constants are defined\n";
} else {
    echo "❌ Missing configuration constants:\n";
    foreach ($missingConstants as $constant) {
        echo "  - $constant\n";
    }
}

echo "\n📊 Summary:\n";
echo "==========\n";
echo "Templates loaded: " . count($templates) . "/22\n";
echo "Template structure: " . (empty($incompleteTemplates) ? "✅ COMPLETE" : "❌ INCOMPLETE") . "\n";
echo "Quick send methods: " . (empty($missingMethods) ? "✅ COMPLETE" : "❌ INCOMPLETE") . "\n";
echo "Configuration: " . (empty($missingConstants) ? "✅ COMPLETE" : "❌ INCOMPLETE") . "\n";

$overallStatus = (count($templates) == 22 && empty($incompleteTemplates) && empty($missingMethods) && empty($missingConstants)) ? "✅ ALL INTEGRATIONS COMPLETE" : "❌ INTEGRATION ISSUES DETECTED";

echo "\n🎉 Overall Integration Status: $overallStatus\n";

// Show sample template data
echo "\n📋 Sample Template Data:\n";
echo "=======================\n";
$templateSample = reset($templates);
$templateName = key($templates);
echo "Template: $templateName\n";
foreach ($templateSample as $key => $value) {
    echo "  $key: " . substr($value, 0, 50) . (strlen($value) > 50 ? "..." : "") . "\n";
}
?>